<?php
// This script checks for upcoming matches (in ~24 hours) and sends reminders for unpaid referee fees.
// It should be set up as a Cron Job to run every hour.
// Example: 0 * * * * /usr/bin/php /path/to/smart_football/app/cron_check_referee_fees.php

// Adjust path to bootstrap if running from CLI vs Web
if (php_sapi_name() === 'cli') {
    require_once __DIR__ . '/../core/bootstrap.php';
} else {
    require_once '../core/bootstrap.php';
}

$db = new Database();
$notificationModel = new Notification();
$teamModel = new Team();
$settingModel = new Setting();

echo "Starting Referee Fee Check...\n";

// Get settings
$settings = $settingModel->getAll();
$currency_symbol = ($settings['system_currency'] ?? 'USD') === 'ZMW' ? 'K' : '$';
$fee_amount = $settings['referee_fee'] ?? 50;

// Calculate time window: 23 to 25 hours from now
// This allows the hourly cron to catch the match once within that window.
$sql = "
    SELECT f.*, 
           c_ht.name as home_team_name, 
           c_at.name as away_team_name
    FROM fixtures f
    JOIN teams ht ON f.home_team_id = ht.id
    JOIN teams at ON f.away_team_id = at.id
    JOIN clubs c_ht ON ht.club_id = c_ht.id
    JOIN clubs c_at ON at.club_id = c_at.id
    WHERE f.match_datetime BETWEEN DATE_ADD(NOW(), INTERVAL 23 HOUR) AND DATE_ADD(NOW(), INTERVAL 25 HOUR) 
    AND f.status = 'scheduled'
";

$db->query($sql);
$fixtures = $db->resultSet();

$count = 0;

foreach ($fixtures as $fixture) {
    $match_time = date('H:i', strtotime($fixture->match_datetime));
    $match_date = date('d M', strtotime($fixture->match_datetime));
    
    // 1. Check Home Team
    if ($fixture->home_team_paid_referee == 0) {
        $team = $teamModel->findById($fixture->home_team_id);
        if ($team && $team->manager_id) {
            $message = "URGENT: Referee fee of {$currency_symbol}{$fee_amount} for the match against " . 
                       $fixture->away_team_name . " on {$match_date} at {$match_time} is UNPAID. " .
                       "Please pay immediately via your Team Dashboard to avoid sanctions.";
            
            $notificationModel->create($team->manager_id, 'fee_reminder', $message);
            if (php_sapi_name() === 'cli') echo "Sent reminder to Home Team Manager (Team ID: {$fixture->home_team_id}) for Fixture #{$fixture->id}\n";
            $count++;
        }
    }

    // 2. Check Away Team
    if ($fixture->away_team_paid_referee == 0) {
        $team = $teamModel->findById($fixture->away_team_id);
        if ($team && $team->manager_id) {
            $message = "URGENT: Referee fee of {$currency_symbol}{$fee_amount} for the match against " . 
                       $fixture->home_team_name . " on {$match_date} at {$match_time} is UNPAID. " .
                       "Please pay immediately via your Team Dashboard to avoid sanctions.";
            
            $notificationModel->create($team->manager_id, 'fee_reminder', $message);
            if (php_sapi_name() === 'cli') echo "Sent reminder to Away Team Manager (Team ID: {$fixture->away_team_id}) for Fixture #{$fixture->id}\n";
            $count++;
        }
    }
}

echo "Done. Sent $count notifications.\n";
?>