<?php
require_once '../core/bootstrap.php';

// --- Authentication and Role Check ---
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'club_manager') {
    header('Location: login.php');
    exit();
}

$pageTitle = 'Payment History';
include 'includes/header.php';

$manager_id = $_SESSION['user_id'];

// --- Instantiate Models ---
$clubManagerModel = new ClubManager();
$settingModel = new Setting();
$db = new Database();

// --- Fetch Data ---
// Get the club managed by the logged-in user
$club = $clubManagerModel->getClubByManagerId($manager_id);

if (!$club) {
    echo "<div class='page-content'><p class='no-data-msg'>You are not currently assigned as a manager to any club.</p></div>";
    include 'includes/footer.php';
    exit();
}

$club_id = $club->id;

// Fetch payments made by this user (manager) or related to the club
// This query assumes payments are linked to the user who made them. 
// For a club manager, this would show payments they initiated (like paying fines).
$db->query("SELECT * FROM payments WHERE user_id = :user_id ORDER BY created_at DESC");
$db->bind(':user_id', $manager_id);
$payments = $db->resultSet();

// --- Get Colors from Settings ---
$settings = $settingModel->getAll();
$primary_color = $settings['primary_color'] ?? '#3498db';
$secondary_color = $settings['secondary_color'] ?? '#2c3e50';
$currency_symbol = ($settings['system_currency'] ?? 'USD') === 'ZMW' ? 'K' : '$';

?>

<div class="team-dashboard-hero">
    <div class="hero-content-wrapper">
        <div class="hero-header-row">
            <div class="team-identity">
                <img src="<?php echo htmlspecialchars($club->logo); ?>" alt="<?php echo htmlspecialchars($club->name); ?>" class="hero-team-logo">
                <div class="team-details">
                    <h1>Payment History</h1>
                    <span class="manager-badge"><?php echo htmlspecialchars($club->name); ?></span>
                </div>
            </div>
            <div class="hero-actions">
                <a href="club_finances.php" class="btn btn-hero-secondary"><i class="fa-solid fa-arrow-left"></i> Back to Finances</a>
            </div>
        </div>
    </div>
</div>

<div class="page-content">
    <div class="dashboard-section">
        <div class="table-wrapper">
            <?php if (!empty($payments)): ?>
                <table class="styled-table">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Transaction Ref</th>
                            <th>Amount</th>
                            <th>Status</th>
                            <th>Details</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($payments as $payment): ?>
                            <tr>
                                <td><?php echo date('d M Y, H:i', strtotime($payment->created_at)); ?></td>
                                <td style="font-family: monospace;"><?php echo htmlspecialchars($payment->transaction_ref); ?></td>
                                <td><strong><?php echo $currency_symbol . number_format($payment->amount, 2); ?></strong></td>
                                <td>
                                    <span class="status-badge status-<?php echo strtolower($payment->status); ?>">
                                        <?php echo ucfirst(htmlspecialchars($payment->status)); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php 
                                    $meta = json_decode($payment->metadata, true);
                                    if (isset($meta['fine_ids'])) {
                                        echo '<span style="font-size: 0.85rem; color: #666;">Fines Paid</span>';
                                    } else {
                                        echo '<span style="font-size: 0.85rem; color: #666;">General Payment</span>';
                                    }
                                    ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <p class="no-data-msg">No payment history found.</p>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
:root {
    --hero-gradient: linear-gradient(135deg, <?php echo $primary_color; ?> 0%, <?php echo $secondary_color; ?> 100%);
}

/* Transparent Header Override */
header {
    background: transparent !important;
    box-shadow: none !important;
    position: absolute !important;
    width: 100%;
    top: 0;
    z-index: 1000;
}

.team-dashboard-hero {
    background: var(--hero-gradient);
    color: white;
    padding: 6rem 20px 3rem 20px;
    border-radius: 0 0 30px 30px;
    margin-bottom: 2rem;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
    margin-top: 0;
}

.status-successful { background: #d4edda; color: #155724; }
.status-failed { background: #f8d7da; color: #721c24; }
.status-pending { background: #fff3cd; color: #856404; }
</style>

<?php include 'includes/footer.php'; ?>