<?php
$pageTitle = 'Suspended Players';
include 'includes/header.php';

// --- Authentication and Role Check ---
if (!isset($_SESSION['user_id']) || $_SESSION['user_role'] !== 'team_manager') {
    header('Location: login.php');
    exit();
}
$manager_id = $_SESSION['user_id'];

// --- Instantiate Models and Fetch Data ---
$teamModel = new Team();
$playerModel = new Player();
$fineModel = new Fine();
$settingModel = new Setting();

// Get the manager's team
$team = $teamModel->getTeamByManagerId($manager_id);

$unavailable_players = [];

// --- Get Currency Symbol from Settings ---
$settings = $settingModel->getAll();
$currency_code = $settings['system_currency'] ?? 'USD';
$currency_symbol = '$';
if ($currency_code === 'ZMW') {
    $currency_symbol = 'K';
} elseif ($currency_code === 'EUR') {
    $currency_symbol = '€';
} // Add other currencies as needed

if ($team) {
    // Get the currently suspended players on the team
    try {
        $suspended_players = $playerModel->getSuspendedPlayersByTeamId($team->id);
    } catch (PDOException $e) {
        // Gracefully handle if 'suspensions' table doesn't exist yet
        $suspended_players = [];
    }
    foreach ($suspended_players as $player) {
        $player->unavailable_reason = 'Match Ban';
        $player->details = $player->reason . ' (' . $player->matches_to_serve . ' matches)';
        $unavailable_players[$player->id] = $player;
    }

    // Get players with unpaid fines
    if (isset($team->club_id)) {
        $outstanding_fines = $fineModel->getUnpaidFinesForTeam($team->club_id);
        foreach ($outstanding_fines as $fine) {
            if ($fine->entity_type === 'player' && !isset($unavailable_players[$fine->player_id])) {
                $fine->unavailable_reason = 'Unpaid Fine';
                $fine->details = $fine->reason . ' (' . $currency_symbol . number_format($fine->amount, 2) . ')';
                $unavailable_players[$fine->player_id] = $fine;
            }
        }
    }
}

?>

<div class="page-content">
    <div class="matches-header">
        <h1>Suspension List</h1>
    </div>

    <?php if ($team): ?>
        <div class="dashboard-section">
            <h2 class="dashboard-title">Unavailable Players</h2>
            <?php if (!empty($unavailable_players)): ?>
                <div class="table-wrapper">
                    <table class="styled-table">
                        <thead><tr><th>Player</th><th>Reason for Unavailability</th><th>Details</th><th>Actions</th></tr></thead>
                        <tbody>
                            <?php foreach ($unavailable_players as $player): ?>
                                <tr>
                                    <td>
                                        <div class="player-cell">
                                            <img src="<?php echo htmlspecialchars($player->profile_picture ?? '/smart_football/public/assets/images/default_avatar.png'); ?>" alt="Player" class="player-avatar-xs">
                                            <span><?php echo htmlspecialchars($player->first_name . ' ' . $player->last_name); ?></span>
                                        </div>
                                    </td>
                                    <td><span class="status-badge status-<?php echo strtolower($player->unavailable_reason) === 'unpaid fine' ? 'fine' : 'suspended'; ?>"><?php echo htmlspecialchars($player->unavailable_reason); ?></span></td>
                                    <td><?php echo htmlspecialchars($player->details); ?></td>
                                    <td>
                                        <?php if (isset($player->suspension_id)): ?>
                                            <span class="text-muted" style="font-size: 0.85rem; color: #777;">No action</span>
                                        <?php elseif (isset($player->entity_type) && $player->entity_type === 'player'): ?>
                                            <a href="my_team.php?tab=fines" class="btn-sm">Pay Fine</a>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <p class="no-data-msg">No players are currently serving a suspension.</p>
            <?php endif; ?>
        </div>
    <?php else: ?>
        <p class="no-data-msg">You are not currently assigned as a manager to any team.</p>
    <?php endif; ?>
</div>

<style>
    .player-cell {
        display: flex;
        align-items: center;
        gap: 12px;
    }
    .player-avatar-xs {
        width: 40px;
        height: 40px;
        border-radius: 50%;
        object-fit: cover;
        flex-shrink: 0;
        border: 1px solid var(--border-color);
    }
</style>

<?php include 'includes/footer.php'; ?>